#!/bin/bash -e

POSITION=1
FILTERS=""
JSON_URL="https://ip-ranges.amazonaws.com/ip-ranges.json"

if [[ -n $1 ]]; then
    POSITION=$1
    shift
fi

build_filters() {
    local filters=""
    for arg in "$@"; do
        if [[ "$arg" == *"*" ]]; then
            local pattern="${arg//\*/.*}"
            pattern="\"$pattern\""
            if [[ -n $filters ]]; then
                filters+=", "
            fi
            filters+="select(.region | test($pattern))"
        else
            if [[ -n $filters ]]; then
                filters+=", "
            fi
            filters+="select(.region | contains(\"$arg\"))"
        fi
    done

    if [[ -n $filters ]]; then
        filters=" | $filters"
    fi

    echo "$filters"
}

extract_ip_ranges() {
    local json=$1
    local filters=$2
    local array=$3
    local prefix=$4

    local group="group_by(.$prefix)"
    local map="map({ \"ip\": .[0].$prefix, \"regions\": map(.region) | unique, \"services\": map(.service) | unique })"
    local to_string='.ip'
    local process="[ .$array[]$filters ] | $group | $map | .[] | $to_string"

    echo "$json" | /usr/bin/jq -r "$process" | sort -Vu
}

create_and_flush_chain() {
    local version=$1
    local position=$2
    local cmd=ip${version}tables

    $cmd -n --list AWS >/dev/null 2>&1 \
        || ($cmd -N AWS && $cmd -I INPUT $position -j AWS)

    $cmd -F AWS
}

add_iptables_rules() {
    local version=$1
    local cmd=ip${version}tables
    local lines
    local data

    IFS=$'\n' lines=($2)
    unset IFS

    for line in "${lines[@]}"; do
        eval local data=($line)
        local ip=${data[0]}
        local regions=$(echo ${data[1]} | tr '[:upper:]' '[:lower:]')
        local services=$(echo ${data[2]} | tr '[:upper:]' '[:lower:]')

        $cmd -A AWS -s "$ip" -j REJECT -m comment --comment "$regions = $services"
    done
}

if [ ! -t 0 ]; then
    JSON=$(cat - <&0)
else
    JSON=$(curl -s -L "$JSON_URL")
fi

echo "Fetched JSON Data (first 20 lines):"
echo "$JSON" | /usr/bin/jq . | head -n 20

if ! echo "$JSON" | /usr/bin/jq empty >/dev/null 2>&1; then
    echo "Invalid JSON data. Exiting."
    exit 1
fi

FILTERS=$(build_filters "$@")
echo "Filters: $FILTERS"

# IPv4
create_and_flush_chain "" $POSITION
V4_RANGES=$(extract_ip_ranges "$JSON" "$FILTERS" "prefixes" "ip_prefix")
echo "IPv4 Ranges:"
echo "$V4_RANGES"

add_iptables_rules "" "$V4_RANGES"

# IPv6
create_and_flush_chain 6 $POSITION
V6_RANGES=$(extract_ip_ranges "$JSON" "$FILTERS" "ipv6_prefixes" "ipv6_prefix")
echo "IPv6 Ranges:"
echo "$V6_RANGES"

add_iptables_rules "6" "$V6_RANGES"

[root@nativespace-vega ~]# ^C
[root@nativespace-vega ~]# cat aws.sh
#!/bin/bash -e

POSITION=1
FILTERS=""
JSON_URL="https://ip-ranges.amazonaws.com/ip-ranges.json"

if [[ -n $1 ]]; then
    POSITION=$1
    shift
fi

build_filters() {
    local filters=""
    for arg in "$@"; do
        if [[ "$arg" == *"*" ]]; then
            local pattern="${arg//\*/.*}"
            pattern="\"$pattern\""
            if [[ -n $filters ]]; then
                filters+=", "
            fi
            filters+="select(.region | test($pattern))"
        else
            if [[ -n $filters ]]; then
                filters+=", "
            fi
            filters+="select(.region | contains(\"$arg\"))"
        fi
    done

    if [[ -n $filters ]]; then
        filters=" | $filters"
    fi

    echo "$filters"
}

extract_ip_ranges() {
    local json=$1
    local filters=$2
    local array=$3
    local prefix=$4

    local group="group_by(.$prefix)"
    local map="map({ \"ip\": .[0].$prefix, \"regions\": map(.region) | unique, \"services\": map(.service) | unique })"
    local to_string='.ip'
    local process="[ .$array[]$filters ] | $group | $map | .[] | $to_string"

    echo "$json" | /usr/bin/jq -r "$process" | sort -Vu
}

create_and_flush_chain() {
    local version=$1
    local position=$2
    local cmd=ip${version}tables

    $cmd -n --list AWS >/dev/null 2>&1 \
        || ($cmd -N AWS && $cmd -I INPUT $position -j AWS)

    $cmd -F AWS
}

add_iptables_rules() {
    local version=$1
    local cmd=ip${version}tables
    local lines
    local data

    IFS=$'\n' lines=($2)
    unset IFS

    for line in "${lines[@]}"; do
        eval local data=($line)
        local ip=${data[0]}
        local regions=$(echo ${data[1]} | tr '[:upper:]' '[:lower:]')
        local services=$(echo ${data[2]} | tr '[:upper:]' '[:lower:]')

        $cmd -A AWS -s "$ip" -j REJECT -m comment --comment "$regions = $services"
    done
}

if [ ! -t 0 ]; then
    JSON=$(cat - <&0)
else
    JSON=$(curl -s -L "$JSON_URL")
fi

echo "Fetched JSON Data (first 20 lines):"
echo "$JSON" | /usr/bin/jq . | head -n 20

if ! echo "$JSON" | /usr/bin/jq empty >/dev/null 2>&1; then
    echo "Invalid JSON data. Exiting."
    exit 1
fi

FILTERS=$(build_filters "$@")
echo "Filters: $FILTERS"

# IPv4
create_and_flush_chain "" $POSITION
V4_RANGES=$(extract_ip_ranges "$JSON" "$FILTERS" "prefixes" "ip_prefix")
echo "IPv4 Ranges:"
echo "$V4_RANGES"

add_iptables_rules "" "$V4_RANGES"

# IPv6
create_and_flush_chain 6 $POSITION
V6_RANGES=$(extract_ip_ranges "$JSON" "$FILTERS" "ipv6_prefixes" "ipv6_prefix")
echo "IPv6 Ranges:"
echo "$V6_RANGES"

add_iptables_rules "6" "$V6_RANGES"
